#include <WiFiS3.h>            // Use WiFiS3 library for Renesas Uno R4 WiFi board
#include <NTPClient.h>
#include <WiFiUdp.h>

// WiFi credentials
const char* ssid = "Your SSID";       // Replace with your WiFi SSID
const char* password = "Your Password";   // Replace with your WiFi password

// ThingSpeak credentials
const char* apiKey = "??????????????";   // Your ThingSpeak Write API Key
const char* server = "api.thingspeak.com";
const int channelID = ???????????;             // Your ThingSpeak Channel ID

// NTP client to get time
WiFiUDP ntpUDP;
NTPClient timeClient(ntpUDP, "pool.ntp.org", 0, 60000); // UTC time, update every minute

// Pin definitions and constants
const int voltagePin = A2;   // Analog pin connected to the voltage divider
const int currentPin = A1;   // Analog pin connected to the ACS758 output
const float R1 = 100000.0;   // Resistance of R1 in ohms (100K)
const float R2 = 10000.0;    // Resistance of R2 in ohms (10K)
const float VCC = 5.0;       // Supply voltage to Arduino (typically 5V)
const float sensitivity = 0.185; // Sensitivity of ACS758 (185 mV/A for 50A version)
float offsetCurrent = 0.07;  // Offset current based on idle reading
float voltageCorrectionFactor = 0.9;  // Adjust based on real voltage readings
float currentCorrectionFactor = 35.9;  // Adjust based on real current readings (calculated from multimeter)

void setup() {
    // Initialize serial communication
    Serial.begin(9600);

    // Attempt to connect to WiFi
    Serial.print("Connecting to ");
    Serial.println(ssid);
    WiFi.begin(ssid, password);
    
    // Wait for WiFi to connect
    while (WiFi.status() != WL_CONNECTED) {
        delay(1000);
        Serial.println("Connecting to WiFi...");
    }
    Serial.println("Connected to WiFi!");

    // Initialize the NTPClient to get time
    timeClient.begin();
}

void loop() {
    // Get the current time
    timeClient.update();
    String formattedTime = timeClient.getFormattedTime();

    // Read the raw analog values from both pins
    int rawValueVoltage = analogRead(voltagePin);
    int rawValueCurrent = analogRead(currentPin);
    
    // Convert the ADC values to voltage (0-1023 corresponds to 0-5V)
    float voltageAtDivider = (rawValueVoltage * VCC) / 1023.0;  // Voltage at the analog pin A0
    float currentVoltage = (rawValueCurrent * VCC) / 1023.0;    // Output voltage from ACS758

    // Calculate the actual system voltage (voltage before the voltage divider)
    float actualVoltage = voltageAtDivider * ((R1 + R2) / R2) * voltageCorrectionFactor;  // Corrected voltage

    // Calculate the current (ACS758 has an offset of VCC/2, which is 2.5V)
    float current = ((currentVoltage - (VCC / 2.0)) / sensitivity) * currentCorrectionFactor;  // Corrected current in amperes
    current = current - offsetCurrent;  // Apply the offset correction

    // Calculate the power in watts
    float power = actualVoltage * current;

    // Display the data in the serial monitor
    Serial.print("Time: ");
    Serial.print(formattedTime);
    Serial.print(" | Voltage: ");
    Serial.print(actualVoltage);
    Serial.print(" V | Current: ");
    Serial.print(current);
    Serial.print(" A | Power: ");
    Serial.print(power);
    Serial.println(" W");

    // Send data to ThingSpeak
    if (WiFi.status() == WL_CONNECTED) { // Check if we're still connected to WiFi
        WiFiClient client;
        
        Serial.println("Connecting to ThingSpeak...");
        if (client.connect(server, 80)) { // Connect to ThingSpeak
            String postStr = String("api_key=") + apiKey +
                             "&field1=" + String(actualVoltage) + 
                             "&field2=" + String(current) + 
                             "&field3=" + String(power);
                             
            // Send HTTP POST request
            client.println("POST /update HTTP/1.1");
            client.println("Host: api.thingspeak.com");
            client.println("Connection: close");
            client.println("Content-Type: application/x-www-form-urlencoded");
            client.println("Content-Length: " + String(postStr.length()));
            client.println();
            client.println(postStr);
            
            Serial.println("POST request sent to ThingSpeak:");
            Serial.println(postStr);

            // Wait for response from ThingSpeak
            while (client.connected() || client.available()) {
                if (client.available()) {
                    String response = client.readString();
                    Serial.println("Response from ThingSpeak: " + response);
                }
            }

            client.stop(); // Close the connection
            Serial.println("Data sent to ThingSpeak.");
        } else {
            Serial.println("Connection to ThingSpeak failed.");
        }
    } else {
        Serial.println("WiFi disconnected, trying to reconnect...");
        WiFi.begin(ssid, password); // Attempt to reconnect to WiFi
    }

    delay(20000); // Wait 20 seconds before the next reading (ThingSpeak has a 15-second rate limit)
}
